#include "mainwindow.h"
#include "./ui_mainwindow.h"
#include "token.h"

#include <QDateTime>
#include <QList>
#include <QErrorMessage>
#include <QByteArray>
#include <QDir>

static const QString INPUT_FILE = "gdb-command-file-list-exec-sources.txt";
static const QString DATE_TIME_FORMAT = "yyyyMMdd-hhmmss.zzz";
static const QString TIME_FORMAT = "hhmmss.zzz";

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::MainWindow)
    , m_inputFile(nullptr)
{
    ui->setupUi(this);

    m_inputFile = new QFile(QDir::home().absoluteFilePath(INPUT_FILE));
    ui->inputFileLineEdit->setText( m_inputFile->fileName());

    connect(ui->startBtn, &QPushButton::clicked, this, &MainWindow::startTest);

}

MainWindow::~MainWindow()
{
    delete ui;
    if (m_inputFile)
    {
        delete m_inputFile;
    }
}

void MainWindow::startTest()
{

    ui->startTimeLineEdit->setText("");
    ui->endTimeLineEdit->setText("");
    ui->elapsedTimeLineEdit->setText("");
    ui->tokensCreatedLineEdit->setText("");

    if (!m_inputFile->exists())
    {
        QErrorMessage errMsg;
        errMsg.showMessage("Input file does not exist!");
        return;
    }

    m_inputFile->open(QIODevice::ReadOnly | QIODevice::Text);

    QByteArray txtArray = m_inputFile->readAll();

    m_inputFile->close();

    if (txtArray.size() < 10)
    {
        QErrorMessage errMsg;
        errMsg.showMessage("Empty input file or unable to read contents");
        return;
    }

    QString fileContents(txtArray);

    QDateTime startTime = QDateTime::currentDateTime();

    ui->startTimeLineEdit->setText(startTime.toString(DATE_TIME_FORMAT));


    QList<Token *> tokenList = tokenize( fileContents);

    QDateTime endTime = QDateTime::currentDateTime();
    ui->endTimeLineEdit->setText(endTime.toString(DATE_TIME_FORMAT));

    QTime elapsedTime = QTime::fromMSecsSinceStartOfDay( startTime.msecsTo(endTime));

    ui->elapsedTimeLineEdit->setText( elapsedTime.toString(TIME_FORMAT));

    QString countStr = QString::number( tokenList.count());
    ui->tokensCreatedLineEdit->setText( countStr);

    // Free tokens
    while ( !tokenList.isEmpty() )
    {
        Token *token = tokenList.takeFirst();
        delete token;
    }

}



/**
 * @brief Creates tokens from a single GDB output row.
 */
QList<Token *> MainWindow::tokenize( QString str )
{
    enum { IDLE, END_CODE, STRING, VAR} state = IDLE;
    QList<Token *> list;
    Token *cur = nullptr;
    bool prevCharIsEscCode = false;

    if ( str.isEmpty() )
    {
        return list;
    }

    for ( int i = 0; i < str.size(); i++ )
    {
        QChar c = str[i];
        bool isEscaped = false;

        if ( c == '\\' && prevCharIsEscCode )
        {
            prevCharIsEscCode = false;
        }
        else if ( prevCharIsEscCode )
        {
            isEscaped = true;
            prevCharIsEscCode = false;
        }
        else if ( c == '\\' )
        {
            prevCharIsEscCode = true;
            continue;
        }

        switch ( state )
        {
            case IDLE:
            {
                if ( c == '"' )
                {
                    cur = new Token( Token::C_STRING );
                    list.push_back( cur );
                    state = STRING;
                }
                else if ( c == '(' )
                {
                    cur = new Token( Token::END_CODE );
                    list.push_back( cur );
                    cur->m_text += c;
                    state = END_CODE;
                }
                else if ( c == '=' || c == '{' || c == '}' || c == ',' ||
                          c == '[' || c == ']' || c == '+' || c == '^' ||
                          c == '~' || c == '@' || c == '&' || c == '*' )
                {
                    Token::Type type = Token::UNKNOWN;

                    if ( c == '=' )
                    {
                        type = Token::KEY_EQUAL;
                    }

                    if ( c == '{' )
                    {
                        type = Token::KEY_LEFT_BRACE;
                    }

                    if ( c == '}' )
                    {
                        type = Token::KEY_RIGHT_BRACE;
                    }

                    if ( c == '[' )
                    {
                        type = Token::KEY_LEFT_BAR;
                    }

                    if ( c == ']' )
                    {
                        type = Token::KEY_RIGHT_BAR;
                    }

                    if ( c == ',' )
                    {
                        type = Token::KEY_COMMA;
                    }

                    if ( c == '^' )
                    {
                        type = Token::KEY_UP;
                    }

                    if ( c == '+' )
                    {
                        type = Token::KEY_PLUS;
                    }

                    if ( c == '~' )
                    {
                        type = Token::KEY_TILDE;
                    }

                    if ( c == '@' )
                    {
                        type = Token::KEY_SNABEL;
                    }

                    if ( c == '&' )
                    {
                        type = Token::KEY_AND;
                    }

                    if ( c == '*' )
                    {
                        type = Token::KEY_STAR;
                    }

                    cur = new Token( type );
                    list.push_back( cur );
                    cur->m_text += c;
                    state = IDLE;
                }
                else if ( c != ' ' )
                {
                    cur = new Token( Token::VAR );
                    list.push_back( cur );
                    cur->m_text = c;
                    state = VAR;
                }

            };

            break;
            case END_CODE:
            {
                QString codeEndStr = "(gdb)";
                cur->m_text += c;

                if ( cur->m_text.length() == codeEndStr.length() )
                {
                    state = IDLE;

                }
                else if ( cur->m_text.compare( codeEndStr.left( cur->m_text.length() ) ) != 0 )
                {
                    cur->setType( Token::VAR );
                    state = IDLE;
                }

            };

            break;
            case STRING:
            {
                if ( c == '"' && isEscaped == false )
                {
                    state = IDLE;
                }
                else if ( isEscaped )
                {
                    if ( c == 'n' )
                    {
                        cur->m_text += '\n';
                    }
                    else if ( c == 't' )
                    {
                        cur->m_text += '\t';
                    }
                    else
                    {
                        cur->m_text += c;
                    }
                }
                else
                {
                    cur->m_text += c;
                }
            };

            break;
            case VAR:
            {
                if ( c == '=' || c == ',' || c == '{' || c == '}' )
                {
                    i--;
                    cur->m_text = cur->m_text.trimmed();
                    state = IDLE;
                }
                else
                {
                    cur->m_text += c;
                }
            };

            break;

        }

    }

    if ( cur )
    {
        if ( cur->getType() == Token::VAR )
        {
            cur->m_text = cur->m_text.trimmed();
        }
    }

    return list;
}
